const express = require("express");
const router = express.Router();
const { check, validationResult } = require("express-validator");
const jwt = require("jsonwebtoken");
const config = require("config");
const multer = require("multer");
const path = require("path");
const fs = require("fs");
const sharp = require("sharp");
const WebSocket = require("ws");

// Import better-sqlite3 models
const User = require("../../models/User");

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = path.join(__dirname, "../../uploads");
    if (!fs.existsSync(uploadDir)) fs.mkdirSync(uploadDir, { recursive: true });
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const ext = path.extname(file.originalname);
    const filename = `avatar-${Date.now()}${ext}`;
    cb(null, filename);
  },
});

const upload = multer({
  storage,
  limits: { fileSize: 80 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const filetypes = /jpeg|jpg|png/;
    const mimetype = filetypes.test(file.mimetype);
    const extname = filetypes.test(
      path.extname(file.originalname).toLowerCase(),
    );

    if (mimetype && extname) return cb(null, true);
    cb(new Error("Only JPEG/JPG/PNG images allowed"));
  },
});

// WebSocket setup
const wss = new WebSocket.Server({ noServer: true });

// Function to broadcast user updates
async function broadcastUserUpdates() {
  try {
    const users = await User.findAll();
    // Remove password from response
    const sanitizedUsers = users.map((u) => {
      const { password, fcmTokens, ...rest } = u;
      return rest;
    });

    const message = JSON.stringify({
      type: "users_update",
      data: sanitizedUsers,
    });

    wss.clients.forEach((client) => {
      if (client.readyState === WebSocket.OPEN) {
        client.send(message);
      }
    });
  } catch (err) {
    console.error("Broadcast error:", err);
  }
}

// @route   POST /api/auth/signup
// @desc    Register new user
// @access  Public
router.post(
  "/signup",
  upload.single("avatar"),
  [
    check("name", "Name is required").trim().notEmpty(),
    check("email", "Valid email required").isEmail().normalizeEmail(),
    check("password", "Password must be 6+ characters").isLength({ min: 6 }),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      if (req.file) fs.unlinkSync(req.file.path);
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, email, password } = req.body;

    try {
      // Check for existing user (case insensitive)
      let user = await User.findByEmail(email);
      if (user) {
        if (req.file) fs.unlinkSync(req.file.path);
        return res
          .status(400)
          .json({ errors: [{ msg: "User already exists" }] });
      }

      // Process avatar with sharp
      let avatar = "";
      if (req.file) {
        const uploadDir = path.join(__dirname, "../../uploads");
        const webpFilename = `avatar-${Date.now()}.webp`;
        const outputPath = path.join(uploadDir, webpFilename);

        await sharp(req.file.path)
          .resize(500, 500, { fit: "cover" })
          .webp({ quality: 80 })
          .toFile(outputPath);

        // remove original
        fs.unlinkSync(req.file.path);

        avatar = `/uploads/${webpFilename}`;
      }

      // Create and save user
      user = await User.create({
        name,
        email: email.toLowerCase(),
        password,
        avatar,
        role: req.body.role || "customer",
      });

      // Broadcast update
      await broadcastUserUpdates();

      // Create JWT
      const payload = {
        user: { id: user.id, role: user.role },
      };

      jwt.sign(
        payload,
        config.get("jwtSecret"),
        { expiresIn: "5 days" },
        (err, token) => {
          if (err) throw err;
          res.json({
            token,
            user: {
              id: user.id,
              name: user.name,
              email: user.email,
              avatar: user.avatar,
              role: user.role,
            },
          });
        },
      );
    } catch (err) {
      console.error("Signup error:", err);
      if (req.file && fs.existsSync(req.file.path))
        fs.unlinkSync(req.file.path);
      res.status(500).send("Server error");
    }
  },
);

// ... (previous code)

// @route   POST /api/auth/login
// @desc    Authenticate user
// @access  Public
router.post("/login", async (req, res) => {
  const { email, password } = req.body;

  if (!email || !password) {
    return res
      .status(400)
      .json({ errors: [{ msg: "Email and password are required" }] });
  }

  try {
    const user = await User.findByEmail(email);
    if (!user) {
      return res.status(400).json({ errors: [{ msg: "Invalid credentials" }] });
    }

    // Compare password
    const isMatch = await User.comparePassword(user, password);
    if (!isMatch) {
      return res.status(400).json({ errors: [{ msg: "Invalid credentials" }] });
    }

    const payload = { user: { id: user.id.toString(), role: user.role } };

    jwt.sign(
      payload,
      config.get("jwtSecret"),
      { expiresIn: "5 days" },
      (err, token) => {
        if (err) throw err;

        res.json({
          token,
          user: {
            id: user.id.toString(),
            name: user.name,
            email: user.email,
            avatar: user.avatar,
            role: user.role,
          },
        });
      },
    );
  } catch (err) {
    console.error("Login error:", err);
    res.status(500).send("Server error");
  }
});

// @route   POST /api/auth/signup
router.post(
  "/signup",
  upload.single("avatar"),
  [
    check("name", "Name is required").trim().notEmpty(),
    check("email", "Valid email required").isEmail().normalizeEmail(),
    check("password", "Password must be 6+ characters").isLength({ min: 6 }),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      if (req.file) fs.unlinkSync(req.file.path);
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, email, password } = req.body;

    try {
      // Check for existing user (case insensitive)
      let user = await User.findByEmail(email);
      if (user) {
        if (req.file) fs.unlinkSync(req.file.path);
        return res
          .status(400)
          .json({ errors: [{ msg: "User already exists" }] });
      }

      // Process avatar with sharp
      let avatar = "";
      if (req.file) {
        const uploadDir = path.join(__dirname, "../../uploads");
        const webpFilename = `avatar-${Date.now()}.webp`;
        const outputPath = path.join(uploadDir, webpFilename);

        await sharp(req.file.path)
          .resize(500, 500, { fit: "cover" })
          .webp({ quality: 80 })
          .toFile(outputPath);

        // remove original
        fs.unlinkSync(req.file.path);

        avatar = `/uploads/${webpFilename}`;
      }

      // Create and save user
      user = await User.create({
        name,
        email: email.toLowerCase(),
        password,
        avatar,
        role: req.body.role || "customer",
      });

      // Broadcast update
      await broadcastUserUpdates();

      // Create JWT
      const payload = {
        user: { id: user.id, role: user.role },
      };

      jwt.sign(
        payload,
        config.get("jwtSecret"),
        { expiresIn: "5 days" },
        (err, token) => {
          if (err) throw err;
          res.json({
            token,
            user: {
              id: user.id,
              name: user.name,
              email: user.email,
              avatar: user.avatar,
              role: user.role,
            },
          });
        },
      );
    } catch (err) {
      console.error("Signup error:", err);
      if (req.file && fs.existsSync(req.file.path))
        fs.unlinkSync(req.file.path);
      res.status(500).send("Server error");
    }
  },
);

// @route   POST /api/auth/save-fcm-token
// @desc    Save FCM token for logged-in user
// @access  Private
router.post("/save-fcm-token", async (req, res) => {
  try {
    const { userId, fcmToken } = req.body;

    if (!userId || !fcmToken) {
      return res
        .status(400)
        .json({ msg: "User ID and FCM token are required" });
    }

    const user = await User.addFcmToken(userId, fcmToken);
    if (!user) return res.status(404).json({ msg: "User not found" });

    res.json({
      msg: "FCM token saved successfully",
      fcmTokens: JSON.parse(user.fcmTokens || "[]"),
    });
  } catch (err) {
    console.error("Save FCM token error:", err);
    res.status(500).send("Server error");
  }
});

// @route   POST /api/auth/email-check
// @desc    Check if email already exists
// @access  Public
router.post(
  "/email-check",
  [check("email", "Valid email required").isEmail().normalizeEmail()],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { email } = req.body;

    try {
      const user = await User.findByEmail(email);

      return res.json({
        exists: !!user,
      });
    } catch (err) {
      console.error("Email check error:", err);
      res.status(500).send("Server error");
    }
  },
);

// @route   GET /api/auth/users
// @desc    Get all users
// @access  Public
router.get("/users", async (req, res) => {
  try {
    const users = await User.findAll();
    // Remove passwords
    const sanitizedUsers = users.map((u) => {
      const { password, fcmTokens, ...rest } = u;
      return rest;
    });
    res.json(sanitizedUsers);
  } catch (err) {
    console.error(err.message);
    res.status(500).send("Server error");
  }
});

// @route   DELETE /api/auth/users/:id
// @desc    Delete user
// @access  Public
router.delete("/users/:id", async (req, res) => {
  try {
    const user = await User.findById(req.params.id);
    if (!user) return res.status(404).json({ msg: "User not found" });

    if (user.avatar) {
      const filePath = path.join(__dirname, "../..", user.avatar);
      if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }

    await User.delete(req.params.id);
    await broadcastUserUpdates();

    res.json({ msg: "User deleted" });
  } catch (err) {
    console.error(err.message);
    res.status(500).send("Server error");
  }
});

// WebSocket upgrade handler
router.ws = (server) => {
  server.on("upgrade", (request, socket, head) => {
    wss.handleUpgrade(request, socket, head, (ws) => {
      wss.emit("connection", ws, request);
    });
  });
};

// @route   POST /api/auth/complete-profile
// @desc    Complete user profile
// @access  Private
router.post("/complete-profile", upload.single("avatar"), async (req, res) => {
  try {
    const { userId, username } = req.body;

    if (!userId || !username) {
      return res.status(400).json({ msg: "Missing fields" });
    }

    let user = await User.findById(userId);
    if (!user) return res.status(404).json({ msg: "User not found" });

    // process avatar
    let avatar = user.avatar;
    if (req.file) {
      const uploadDir = path.join(__dirname, "../../uploads");
      const webpFilename = `avatar-${Date.now()}.webp`;
      const outputPath = path.join(uploadDir, webpFilename);

      await sharp(req.file.path)
        .resize(500, 500, { fit: "cover" })
        .webp({ quality: 80 })
        .toFile(outputPath);

      fs.unlinkSync(req.file.path);
      avatar = `/uploads/${webpFilename}`;
    }

    user = await User.update(userId, {
      name: username,
      avatar,
    });

    res.json({
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
        avatar: user.avatar,
        role: user.role,
      },
    });
  } catch (err) {
    console.error(err);
    res.status(500).send("Server error");
  }
});

module.exports = router;
