const express = require("express");
const router = express.Router();

// Import better-sqlite3 models directly
const Duty = require("../../models/Duty");
const User = require("../../models/User");

const adminController = require("../../controllers/admin.controller");

// Teachers
router.get("/teachers", adminController.getAllTeachers);

// Students
router.get("/students", adminController.getAllStudents);
router.put("/student/:studentId", adminController.updateStudent);
router.delete("/student/:studentId", adminController.deleteStudent);
router.get("/students/list", adminController.getAllStudentsList);
router.get("/teachers-for-duty", adminController.getTeachersForDuty);

// Duties
router.post("/assign-duty", adminController.assignDutyToTeacher);

/* ================= DUTY ROUTES ================= */

// Get all duties (admin)
router.get("/duties/all", async (req, res) => {
  try {
    const duties = Duty.findAllWithTeacher();

    const formattedDuties = duties.map((duty) => ({
      id: duty.id,
      teacherId: duty.teacherId,
      teacherName: duty.teacher_name,
      title: duty.title,
      description: duty.description,
      dueDate: duty.dueDate,
      isCompleted: Boolean(duty.isCompleted),
      createdAt: duty.created_at,
      updatedAt: duty.updated_at,
    }));

    res.status(200).json({
      success: true,
      duties: formattedDuties,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
});

// Get teacher duties
router.get("/duties/teacher/:teacherId", async (req, res) => {
  try {
    const { teacherId } = req.params;

    const duties = Duty.findByTeacher(teacherId);

    const formattedDuties = duties.map((duty) => ({
      id: duty.id,
      teacherId: duty.teacherId,
      teacherName: duty.teacher_name,
      title: duty.title,
      description: duty.description,
      dueDate: duty.dueDate,
      isCompleted: Boolean(duty.isCompleted),
      createdAt: duty.created_at,
      updatedAt: duty.updated_at,
    }));

    res.status(200).json({
      success: true,
      duties: formattedDuties,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
});

// Update duty status
router.put("/duties/:dutyId/status", async (req, res) => {
  try {
    const { dutyId } = req.params;
    const { isCompleted } = req.body;

    const duty = await Duty.findById(dutyId);
    if (!duty) {
      return res.status(404).json({
        success: false,
        message: "Duty not found",
      });
    }

    const updatedDuty = Duty.update(dutyId, {
      isCompleted,
      status: isCompleted ? "completed" : "pending",
    });

    // Get teacher info
    const teacher = await User.findById(duty.teacher_id);

    res.status(200).json({
      success: true,
      message: "Duty status updated",
      duty: {
        id: updatedDuty.id,
        teacherId: updatedDuty.teacherId,
        teacherName: teacher?.name,
        title: updatedDuty.title,
        description: updatedDuty.description,
        dueDate: updatedDuty.dueDate,
        isCompleted: Boolean(updatedDuty.isCompleted),
        createdAt: updatedDuty.created_at,
        updatedAt: updatedDuty.updated_at,
      },
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
});

// Delete duty (admin only)
router.delete("/duties/:dutyId", async (req, res) => {
  try {
    const { dutyId } = req.params;

    const duty = Duty.delete(dutyId);
    if (!duty) {
      return res.status(404).json({
        success: false,
        message: "Duty not found",
      });
    }

    res.status(200).json({
      success: true,
      message: "Duty deleted successfully",
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
});

module.exports = router;
