const { get, run, query } = require("../config/db");

const SubjectRating = {
  // Helper to convert rating object with string IDs
  toSafeObject: (rating) => {
    if (!rating) return null;
    return {
      id: rating.id.toString(),
      studentId: rating.student_id?.toString(),
      subject: rating.subject,
      date: rating.date,
      rating: rating.rating,
      created_at: rating.created_at,
      updated_at: rating.updated_at,
    };
  },

  // Create or update rating
  upsert: (ratingData) => {
    const { studentId, subject, date, rating } = ratingData;

    const existing = get(
      "SELECT * FROM subject_ratings WHERE student_id = ? AND subject = ? AND date = ?",
      [studentId, subject, date],
    );

    if (existing) {
      run("UPDATE subject_ratings SET rating = ? WHERE id = ?", [
        rating,
        existing.id,
      ]);
      const updated = SubjectRating.findById(existing.id);
      return SubjectRating.toSafeObject(updated);
    } else {
      const result = run(
        "INSERT INTO subject_ratings (student_id, subject, date, rating) VALUES (?, ?, ?, ?)",
        [studentId, subject, date, rating],
      );
      const created = SubjectRating.findById(result.lastInsertRowid);
      return SubjectRating.toSafeObject(created);
    }
  },

  // Find by ID
  findById: (id) => {
    return get("SELECT * FROM subject_ratings WHERE id = ?", [id]);
  },

  // Find by student
  findByStudent: (studentId) => {
    const ratings = query(
      "SELECT * FROM subject_ratings WHERE student_id = ? ORDER BY date DESC",
      [studentId],
    );
    return ratings.map((r) => SubjectRating.toSafeObject(r));
  },

  // Get ratings grouped by subject for a student
  getGroupedByStudent: (studentId) => {
    const ratings = query(
      "SELECT subject, date, rating FROM subject_ratings WHERE student_id = ? ORDER BY date DESC",
      [studentId],
    );

    const grouped = {};
    ratings.forEach((r) => {
      if (!grouped[r.subject]) grouped[r.subject] = {};
      grouped[r.subject][r.date] = r.rating;
    });

    return grouped;
  },

  // Delete rating
  delete: (id) => {
    const rating = SubjectRating.findById(id);
    run("DELETE FROM subject_ratings WHERE id = ?", [id]);
    return SubjectRating.toSafeObject(rating);
  },
};

module.exports = SubjectRating;
