// models/Student.js
const { get, run, query } = require("../config/db");

const Student = {
  // Helper to convert student object with string IDs
  toSafeObject: (student) => {
    if (!student) return null;
    return {
      id: student.id.toString(),
      name: student.name,
      age: student.age,
      phone: student.phone,
      fatherName: student.fatherName,
      description: student.description,
      teacherId: student.teacher_id?.toString(),
      photo: student.photo,
      photoWebP: student.photo_webp,
      created_at: student.created_at,
      updated_at: student.updated_at,
    };
  },

  // Create a new student
  create: (studentData) => {
    const {
      name,
      age,
      phone,
      fatherName,
      description,
      teacherId,
      photo,
      photoWebP,
    } = studentData;

    const result = run(
      `INSERT INTO students (name, age, phone, fatherName, description, teacher_id, photo, photo_webp)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        name,
        age,
        phone,
        fatherName,
        description || "",
        teacherId,
        photo,
        photoWebP,
      ],
    );

    const student = Student.findById(result.lastInsertRowid);
    return Student.toSafeObject(student);
  },

  // Find student by ID
  findById: (id) => {
    return get("SELECT * FROM students WHERE id = ?", [id]);
  },

  // Find all students with pagination and search
  findAll: (options = {}) => {
    let sql = "SELECT * FROM students";
    const params = [];
    const conditions = [];

    if (options.where) {
      if (options.where.teacherId) {
        conditions.push("teacher_id = ?");
        params.push(options.where.teacherId);
      }

      if (options.where.search) {
        conditions.push("(name LIKE ? OR fatherName LIKE ?)");
        params.push(`%${options.where.search}%`, `%${options.where.search}%`);
      }
    }

    if (conditions.length > 0) {
      sql += " WHERE " + conditions.join(" AND ");
    }

    // Get total count
    const countSql = sql.replace("*", "COUNT(*) as total");
    const countResult = get(countSql, params);
    const total = countResult ? countResult.total : 0;

    // Add pagination
    if (options.limit) {
      sql += " ORDER BY created_at DESC LIMIT ? OFFSET ?";
      params.push(options.limit, options.offset || 0);
    }

    const students = query(sql, params);

    return {
      rows: students.map((s) => Student.toSafeObject(s)),
      count: total,
    };
  },

  // Find students by teacher with pagination
  findByTeacher: (teacherId, options = {}) => {
    const limit = options.limit || 10;
    const offset = options.offset || 0;

    const students = query(
      "SELECT * FROM students WHERE teacher_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?",
      [teacherId, limit, offset],
    );

    const countResult = get(
      "SELECT COUNT(*) as total FROM students WHERE teacher_id = ?",
      [teacherId],
    );

    return {
      rows: students.map((s) => Student.toSafeObject(s)),
      count: countResult.total,
    };
  },

  // Update student
  update: (id, updates) => {
    const fields = [];
    const values = [];

    const fieldMap = {
      teacherId: "teacher_id",
      photoWebP: "photo_webp",
      fatherName: "fatherName", // Add this mapping
    };

    Object.entries(updates).forEach(([key, value]) => {
      if (value !== undefined) {
        const dbField = fieldMap[key] || key;
        fields.push(`${dbField} = ?`);
        values.push(value);
      }
    });

    if (fields.length === 0) return null;

    values.push(id);
    run(`UPDATE students SET ${fields.join(", ")} WHERE id = ?`, values);

    const student = Student.findById(id);
    return Student.toSafeObject(student);
  },

  // Delete student
  delete: (id) => {
    const student = Student.findById(id);
    run("DELETE FROM students WHERE id = ?", [id]);
    return Student.toSafeObject(student);
  },

  // Get students with teacher info (for admin list)
  getAllWithTeacherInfo: () => {
    const students = query(`
      SELECT s.*, u.name as teacher_name, u.email as teacher_email
      FROM students s
      LEFT JOIN users u ON s.teacher_id = u.id
      ORDER BY s.created_at DESC
    `);

    return students.map((s) => ({
      id: s.id.toString(),
      name: s.name,
      age: s.age,
      phone: s.phone,
      fatherName: s.fatherName,
      description: s.description,
      photo: s.photo,
      photo_webp: s.photo_webp,
      teacher_id: s.teacher_id?.toString(),
      teacher_name: s.teacher_name,
      teacher_email: s.teacher_email,
      created_at: s.created_at,
      updated_at: s.updated_at,
    }));
  },

  // Get student count by teacher
  getCountByTeacher: (teacherId) => {
    const result = get(
      "SELECT COUNT(*) as count FROM students WHERE teacher_id = ?",
      [teacherId],
    );
    return result ? result.count : 0;
  },

  // Search students by name or fatherName
  search: (searchTerm, options = {}) => {
    const limit = options.limit || 20;
    const offset = options.offset || 0;

    const students = query(
      `SELECT * FROM students
       WHERE name LIKE ? OR fatherName LIKE ?
       ORDER BY created_at DESC
       LIMIT ? OFFSET ?`,
      [`%${searchTerm}%`, `%${searchTerm}%`, limit, offset],
    );

    const countResult = get(
      `SELECT COUNT(*) as total FROM students
       WHERE name LIKE ? OR fatherName LIKE ?`,
      [`%${searchTerm}%`, `%${searchTerm}%`],
    );

    return {
      rows: students.map((s) => Student.toSafeObject(s)),
      count: countResult.total,
    };
  },

  // Batch create students (for migrations)
  batchCreate: (studentsData) => {
    const insertMany = db.transaction((students) => {
      for (const student of students) {
        Student.create(student);
      }
    });

    return insertMany(studentsData);
  },

  // Update multiple students (e.g., reassign teacher)
  updateTeacher: (studentIds, newTeacherId) => {
    const placeholders = studentIds.map(() => "?").join(",");
    run(`UPDATE students SET teacher_id = ? WHERE id IN (${placeholders})`, [
      newTeacherId,
      ...studentIds,
    ]);

    return studentIds.length;
  },

  // Get students without teacher
  getUnassigned: (options = {}) => {
    const limit = options.limit || 50;
    const offset = options.offset || 0;

    const students = query(
      'SELECT * FROM students WHERE teacher_id IS NULL OR teacher_id = "" ORDER BY created_at DESC LIMIT ? OFFSET ?',
      [limit, offset],
    );

    const countResult = get(
      'SELECT COUNT(*) as total FROM students WHERE teacher_id IS NULL OR teacher_id = ""',
    );

    return {
      rows: students.map((s) => Student.toSafeObject(s)),
      count: countResult.total,
    };
  },

  // Get recent students
  getRecent: (limit = 10) => {
    const students = query(
      "SELECT * FROM students ORDER BY created_at DESC LIMIT ?",
      [limit],
    );
    return students.map((s) => Student.toSafeObject(s));
  },

  // Check if student exists
  exists: (id) => {
    const result = get("SELECT 1 FROM students WHERE id = ?", [id]);
    return !!result;
  },

  // Get students by multiple IDs
  findByIds: (ids) => {
    if (!ids.length) return [];

    const placeholders = ids.map(() => "?").join(",");
    const students = query(
      `SELECT * FROM students WHERE id IN (${placeholders})`,
      ids,
    );
    return students.map((s) => Student.toSafeObject(s));
  },
};

module.exports = Student;
