const { get, run, query } = require("../config/db");

const Duty = {
  // Helper to convert duty object with string IDs
  toSafeObject: (duty) => {
    if (!duty) return null;
    return {
      id: duty.id.toString(),
      teacherId: duty.teacher_id?.toString(),
      title: duty.title,
      description: duty.description,
      dueDate: duty.due_date,
      status: duty.status,
      isCompleted: Boolean(duty.is_completed),
      created_at: duty.created_at,
      updated_at: duty.updated_at,
    };
  },

  // Create duty
  create: (dutyData) => {
    const {
      teacherId,
      title,
      description,
      dueDate,
      status = "pending",
      isCompleted = false,
    } = dutyData;

    const result = run(
      `INSERT INTO duties (teacher_id, title, description, due_date, status, is_completed)
       VALUES (?, ?, ?, ?, ?, ?)`,
      [teacherId, title, description, dueDate, status, isCompleted ? 1 : 0],
    );

    const duty = Duty.findById(result.lastInsertRowid);
    return Duty.toSafeObject(duty);
  },

  // Find by ID
  findById: (id) => {
    return get("SELECT * FROM duties WHERE id = ?", [id]);
  },

  // Find by teacher
  findByTeacher: (teacherId, options = {}) => {
    let sql =
      "SELECT d.*, u.name as teacher_name, u.email as teacher_email FROM duties d LEFT JOIN users u ON d.teacher_id = u.id WHERE d.teacher_id = ?";
    const params = [teacherId];

    if (options.status) {
      sql += " AND d.status = ?";
      params.push(options.status);
    }

    if (options.startDate && options.endDate) {
      sql += " AND d.due_date BETWEEN ? AND ?";
      params.push(options.startDate, options.endDate);
    }

    sql += " ORDER BY d.due_date ASC";

    const duties = query(sql, params);
    return duties.map((d) => ({
      ...Duty.toSafeObject(d),
      teacher_name: d.teacher_name,
      teacher_email: d.teacher_email,
    }));
  },

  // Find all with teacher info
  findAllWithTeacher: (options = {}) => {
    let sql =
      "SELECT d.*, u.name as teacher_name, u.email as teacher_email FROM duties d LEFT JOIN users u ON d.teacher_id = u.id";
    const params = [];
    const conditions = [];

    if (options.status) {
      conditions.push("d.status = ?");
      params.push(options.status);
    }

    if (options.teacherId) {
      conditions.push("d.teacher_id = ?");
      params.push(options.teacherId);
    }

    if (conditions.length > 0) {
      sql += " WHERE " + conditions.join(" AND ");
    }

    sql += " ORDER BY d.due_date ASC";

    const duties = query(sql, params);
    return duties.map((d) => ({
      ...Duty.toSafeObject(d),
      teacher_name: d.teacher_name,
      teacher_email: d.teacher_email,
    }));
  },

  // Update duty
  update: (id, updates) => {
    const fields = [];
    const values = [];

    const fieldMap = {
      teacherId: "teacher_id",
      dueDate: "due_date",
      isCompleted: "is_completed",
    };

    Object.entries(updates).forEach(([key, value]) => {
      if (value !== undefined) {
        const dbField = fieldMap[key] || key;
        fields.push(`${dbField} = ?`);
        values.push(key === "isCompleted" ? (value ? 1 : 0) : value);
      }
    });

    if (fields.length === 0) return null;

    values.push(id);
    run(`UPDATE duties SET ${fields.join(", ")} WHERE id = ?`, values);

    const duty = Duty.findById(id);
    return Duty.toSafeObject(duty);
  },

  // Delete duty
  delete: (id) => {
    const duty = Duty.findById(id);
    run("DELETE FROM duties WHERE id = ?", [id]);
    return Duty.toSafeObject(duty);
  },

  // Get statistics for a teacher
  getStatistics: (teacherId) => {
    const stats = get(
      `
      SELECT
        COUNT(*) as total,
        SUM(CASE WHEN is_completed = 1 THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN is_completed = 0 THEN 1 ELSE 0 END) as pending,
        SUM(CASE
          WHEN due_date >= date('now')
          AND due_date <= date('now', '+7 days')
          AND is_completed = 0
          THEN 1 ELSE 0
        END) as upcoming
      FROM duties
      WHERE teacher_id = ?
    `,
      [teacherId],
    );

    return {
      total: stats.total || 0,
      completed: stats.completed || 0,
      pending: stats.pending || 0,
      upcoming: stats.upcoming || 0,
      completionRate:
        stats.total > 0 ? Math.round((stats.completed / stats.total) * 100) : 0,
    };
  },

  // Check if duty exists for teacher on date
  existsForTeacherOnDate: (teacherId, dueDate) => {
    const result = get(
      "SELECT id FROM duties WHERE teacher_id = ? AND due_date = ? AND status = ?",
      [teacherId, dueDate, "pending"],
    );
    return !!result;
  },
};

module.exports = Duty;
