const { get, run, query } = require("../config/db");

const Attendance = {
  // Helper to convert attendance object with string IDs
  toSafeObject: (attendance) => {
    if (!attendance) return null;
    return {
      id: attendance.id.toString(),
      studentId: attendance.student_id?.toString(),
      date: attendance.date,
      isPresent: Boolean(attendance.is_present),
      markedBy: attendance.marked_by?.toString(),
      created_at: attendance.created_at,
      updated_at: attendance.updated_at,
    };
  },

  // Create or update attendance
  upsert: (attendanceData) => {
    const { studentId, date, isPresent, markedBy } = attendanceData;

    const existing = get(
      "SELECT * FROM attendances WHERE student_id = ? AND date = ?",
      [studentId, date],
    );

    if (existing) {
      run("UPDATE attendances SET is_present = ?, marked_by = ? WHERE id = ?", [
        isPresent,
        markedBy,
        existing.id,
      ]);
      const attendance = Attendance.findById(existing.id);
      return Attendance.toSafeObject(attendance);
    } else {
      const result = run(
        "INSERT INTO attendances (student_id, date, is_present, marked_by) VALUES (?, ?, ?, ?)",
        [studentId, date, isPresent, markedBy],
      );
      const attendance = Attendance.findById(result.lastInsertRowid);
      return Attendance.toSafeObject(attendance);
    }
  },

  // Find by ID
  findById: (id) => {
    return get("SELECT * FROM attendances WHERE id = ?", [id]);
  },

  // Find by student ID with optional date range
  findByStudent: (studentId, options = {}) => {
    let sql = "SELECT * FROM attendances WHERE student_id = ?";
    const params = [studentId];

    if (options.startDate && options.endDate) {
      sql += " AND date BETWEEN ? AND ?";
      params.push(options.startDate, options.endDate);
    }

    sql += " ORDER BY date ASC";

    const attendances = query(sql, params);
    return attendances.map((a) => Attendance.toSafeObject(a));
  },

  // Get attendance for a specific date
  findByDate: (date) => {
    const attendances = query("SELECT * FROM attendances WHERE date = ?", [
      date,
    ]);
    return attendances.map((a) => Attendance.toSafeObject(a));
  },

  // Delete attendance
  delete: (id) => {
    const attendance = Attendance.findById(id);
    run("DELETE FROM attendances WHERE id = ?", [id]);
    return Attendance.toSafeObject(attendance);
  },
};

module.exports = Attendance;
