const Duty = require("../models/Duty");
const User = require("../models/User");

/* =========================================================
   GET TEACHER'S DUTIES
========================================================= */

exports.getTeacherDuties = async (req, res) => {
  try {
    const teacherId = req.params.teacherId || req.user.id;

    const duties = await Duty.findByTeacher(teacherId);

    res.status(200).json({
      success: true,
      duties,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   GET ALL DUTIES (ADMIN)
========================================================= */
exports.getAllDuties = async (req, res) => {
  try {
    const duties = await Duty.findAllWithTeacher();

    res.status(200).json({
      success: true,
      duties,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   UPDATE DUTY STATUS
========================================================= */
exports.updateDutyStatus = async (req, res) => {
  try {
    const { dutyId } = req.params;
    const { isCompleted } = req.body;

    const duty = await Duty.findById(dutyId);
    if (!duty) {
      return res.status(404).json({
        success: false,
        message: "Duty not found",
      });
    }

    const updatedDuty = await Duty.update(dutyId, {
      isCompleted,
      status: isCompleted ? "completed" : "pending",
    });

    // Get teacher info
    const teacher = await User.findById(duty.teacher_id);

    res.status(200).json({
      success: true,
      message: "Duty status updated",
      duty: {
        ...updatedDuty,
        teacherName: teacher?.name,
      },
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   DELETE DUTY (ADMIN ONLY)
========================================================= */
exports.deleteDuty = async (req, res) => {
  try {
    const { dutyId } = req.params;

    const duty = await Duty.delete(dutyId);
    if (!duty) {
      return res.status(404).json({
        success: false,
        message: "Duty not found",
      });
    }

    res.status(200).json({
      success: true,
      message: "Duty deleted successfully",
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   GET DUTY STATISTICS
========================================================= */
exports.getDutyStatistics = async (req, res) => {
  try {
    const teacherId = req.query.teacherId || req.user.id;

    const statistics = await Duty.getStatistics(teacherId);

    res.status(200).json({
      success: true,
      statistics,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   GET DUTIES BY DATE RANGE
========================================================= */
exports.getDutiesByDateRange = async (req, res) => {
  try {
    const teacherId = req.query.teacherId || req.user.id;
    const { startDate, endDate } = req.query;

    const duties = await Duty.findByTeacher(teacherId, {
      startDate,
      endDate,
    });

    res.status(200).json({
      success: true,
      duties,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};
console.log("Controller version 2 loaded");
