const User = require("../models/User");
const Student = require("../models/Student");
const Duty = require("../models/Duty");

/* =========================================================
   10️⃣ ADMIN AUTH CHECK
========================================================= */
exports.adminOnly = (req, res, next) => {
  if (!req.user || req.user.role !== "admin") {
    return res.status(403).json({ message: "Admin access only" });
  }
  next();
};

/* =========================================================
   1️⃣ GET ALL TEACHERS
========================================================= */
exports.getAllTeachers = async (req, res) => {
  try {
    const teachers = User.findAll({ where: { role: "teacher" } });
    res.status(200).json(teachers);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   2️⃣ GET STUDENTS OF A SPECIFIC TEACHER
========================================================= */
exports.getTeacherStudents = async (req, res) => {
  try {
    const { teacher_id } = req.params;
    const students = Student.findAll({ where: { teacher_id } });
    res.status(200).json(students);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   3️⃣ GET ALL STUDENTS (ADMIN VIEW)
========================================================= */
exports.getAllStudents = async (req, res) => {
  try {
    const students = Student.getAllWithTeacherInfo();
    res.status(200).json(students);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   4️⃣ ASSIGN DUTY TO TEACHER
========================================================= */
exports.assignDutyToTeacher = async (req, res) => {
  try {
    const { teacherId, title, description, dueDate } = req.body;

    if (!teacherId || !title || !dueDate) {
      return res.status(400).json({ message: "Missing required fields" });
    }

    // Check if teacher exists
    const teacher = await User.findById(teacherId);
    if (!teacher || teacher.role !== "teacher") {
      return res.status(404).json({ message: "Teacher not found" });
    }

    // prevent duplicate duty on same date
    const exists = Duty.existsForTeacherOnDate(teacherId, dueDate);

    if (exists) {
      return res
        .status(409)
        .json({ message: "Duty already assigned for this date" });
    }

    const duty = Duty.create({
      teacherId,
      title,
      description,
      dueDate,
      status: "pending",
      isCompleted: false,
    });

    res.status(201).json({
      message: "Duty assigned successfully",
      duty,
    });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   GET TEACHERS FOR DUTY ASSIGNMENT
========================================================= */
exports.getTeachersForDuty = async (req, res) => {
  try {
    const teachers = User.getTeachers();

    const simplifiedTeachers = teachers.map((teacher) => ({
      id: teacher.id.toString(),
      name: teacher.name,
      email: teacher.email,
      displayName: teacher.name,
      role: "teacher",
    }));

    res.status(200).json(simplifiedTeachers);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   5️⃣ UPDATE STUDENT (ADMIN)
========================================================= */
exports.updateStudent = async (req, res) => {
  try {
    const { studentId } = req.params;

    const student = await Student.findById(studentId);
    if (!student) {
      return res.status(404).json({ message: "Student not found" });
    }

    const updatedStudent = await Student.update(studentId, req.body);

    res.status(200).json({
      message: "Student updated successfully",
      student: updatedStudent,
    });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

/* =========================================================
   📋 GET ALL STUDENTS (WITH TEACHER INFO)
========================================================= */
exports.getAllStudentsList = async (req, res) => {
  try {
    const students = Student.getAllWithTeacherInfo();

    const formattedStudents = students.map((student) => ({
      id: student.id,
      name: student.name,
      age: student.age,
      phone: student.phone,
      fatherName: student.fatherName,
      description: student.description,
      photo: student.photo,
      photoWebP: student.photo_webp,
      teacherId: student.teacher_id,
      teacherName: student.teacher_name || "Not Assigned",
      teacherEmail: student.teacher_email || null,
      createdAt: student.created_at,
      updatedAt: student.updated_at,
    }));

    res.status(200).json({
      success: true,
      count: formattedStudents.length,
      students: formattedStudents,
    });
  } catch (err) {
    res.status(500).json({
      success: false,
      message: err.message,
    });
  }
};

/* =========================================================
   DELETE STUDENT (ADMIN)
========================================================= */
exports.deleteStudent = async (req, res) => {
  try {
    const { studentId } = req.params;

    const student = Student.delete(studentId);
    if (!student) {
      return res.status(404).json({ message: "Student not found" });
    }

    res.status(200).json({ message: "Student deleted successfully" });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};
