const Database = require("better-sqlite3");
const path = require("path");
const fs = require("fs");

// Ensure database directory exists
const dbDir = path.join(__dirname, "..", "database");
if (!fs.existsSync(dbDir)) {
  fs.mkdirSync(dbDir, { recursive: true });
}

const dbPath = path.join(dbDir, "jamia.sqlite");

// Initialize database with verbose mode for debugging
const db = new Database(dbPath, {
  verbose: process.env.NODE_ENV === "development" ? console.log : null,
  fileMustExist: false,
});

// Enable foreign keys
db.pragma("foreign_keys = ON");

// Initialize database tables
const initDB = () => {
  // Create tables if they don't exist
  db.exec(`
    -- Users table
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      email TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      avatar TEXT,
      role TEXT DEFAULT 'customer',
      fcmTokens TEXT DEFAULT '[]',
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    );

    -- Students table
    CREATE TABLE IF NOT EXISTS students (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      age INTEGER NOT NULL,
      phone TEXT NOT NULL,
      fatherName TEXT NOT NULL,
      description TEXT DEFAULT '',
      teacher_id TEXT NOT NULL,
      photo TEXT,
      photo_webp TEXT,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    );

    -- Duties table
    CREATE TABLE IF NOT EXISTS duties (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      teacher_id INTEGER NOT NULL,
      title TEXT,
      description TEXT,
      due_date DATE,
      status TEXT DEFAULT 'pending',
      is_completed BOOLEAN DEFAULT 0,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (teacher_id) REFERENCES users(id) ON DELETE CASCADE
    );

    -- Attendances table
    CREATE TABLE IF NOT EXISTS attendances (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      student_id INTEGER NOT NULL,
      date DATE NOT NULL,
      is_present BOOLEAN NOT NULL,
      marked_by INTEGER,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
      FOREIGN KEY (marked_by) REFERENCES users(id) ON DELETE SET NULL,
      UNIQUE(student_id, date)
    );

    -- Subject ratings table
    CREATE TABLE IF NOT EXISTS subject_ratings (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      student_id INTEGER NOT NULL,
      subject TEXT NOT NULL,
      date DATE NOT NULL,
      rating TEXT CHECK( rating IN ('Perfect','Good','Bad') ) NOT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
      UNIQUE(student_id, subject, date)
    );

    -- Create indexes for better performance
    CREATE INDEX IF NOT EXISTS idx_students_teacher_id ON students(teacher_id);
    CREATE INDEX IF NOT EXISTS idx_duties_teacher_due_date ON duties(teacher_id, due_date);
    CREATE INDEX IF NOT EXISTS idx_attendances_student_date ON attendances(student_id, date);
    CREATE INDEX IF NOT EXISTS idx_ratings_student ON subject_ratings(student_id);
  `);

  // Create triggers for updated_at
  db.exec(`
    CREATE TRIGGER IF NOT EXISTS update_users_updated_at
    AFTER UPDATE ON users
    BEGIN
      UPDATE users SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
    END;

    CREATE TRIGGER IF NOT EXISTS update_students_updated_at
    AFTER UPDATE ON students
    BEGIN
      UPDATE students SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
    END;

    CREATE TRIGGER IF NOT EXISTS update_duties_updated_at
    AFTER UPDATE ON duties
    BEGIN
      UPDATE duties SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
    END;

    CREATE TRIGGER IF NOT EXISTS update_attendances_updated_at
    AFTER UPDATE ON attendances
    BEGIN
      UPDATE attendances SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
    END;

    CREATE TRIGGER IF NOT EXISTS update_ratings_updated_at
    AFTER UPDATE ON subject_ratings
    BEGIN
      UPDATE subject_ratings SET updated_at = CURRENT_TIMESTAMP WHERE id = NEW.id;
    END;
  `);
};

// Initialize tables
initDB();

// Helper function to run queries with parameters
const query = (sql, params = []) => {
  try {
    const stmt = db.prepare(sql);
    return stmt.all(params);
  } catch (error) {
    console.error("Query error:", error);
    throw error;
  }
};

// Helper function to get a single row
const get = (sql, params = []) => {
  try {
    const stmt = db.prepare(sql);
    return stmt.get(params);
  } catch (error) {
    console.error("Get error:", error);
    throw error;
  }
};

// Helper function to run insert/update/delete
const run = (sql, params = []) => {
  try {
    const stmt = db.prepare(sql);
    return stmt.run(params);
  } catch (error) {
    console.error("Run error:", error);
    throw error;
  }
};

// Helper function for transactions
const transaction = (fn) => {
  return db.transaction(fn);
};

const connectDB = () => {
  try {
    // Test the connection
    db.prepare("SELECT 1").get();
    console.log("✅ SQLite Connected...");
    return db;
  } catch (err) {
    console.error("❌ SQLite Connection Error:", err.message);
    process.exit(1);
  }
};

module.exports = {
  db,
  query,
  get,
  run,
  transaction,
  connectDB,
};
